/*********************Stata code generating Stata file gperc_fiscalincome_1900_2014.dta (long-run fiscal income series generated from income tax tabulations), as well as extracted tables TableD10.dta, TableD11.dta, etc, using input from SimulationMatLab and IncomeTaxTabulations*********/


clear
set more off

* Project directory
	* John
		global projdir "C:\Users\John\Dropbox\WIDFrance\Papers\GGP2017DINA"
	* Bertrand 
		*global projdir "C:\Users\G839276\Dropbox\WIDFrance\Papers\GGP2017DINA"
	* Thomas
		*global projdir "WIDFrance\Papers\GGP2017DINA"


/**********************importing average income and correction factors 1900-2014****************/
		cd "$projdir/GGP2017DINAAppendixD/"
		clear
		import excel "IncomeTaxTabulations\GGP2017IncomeTaxAverages.xlsx", sheet("Averages") firstrow
		keep if (year>=1915 & year<=2014) | year==1900 | year==1910
		save StataFiles\temp\averages.dta, replace
		clear

/*******************importing gperc indiv series for taxable income 1900-2014******************/
	* 1900-2013
		clear
		foreach year in 1900 1910 {
			import delimited StataFiles/SimulationMatlab/output/indiv_bysheet/gperc_indiv_`year'.csv
			gen year=`year'
			if `year'>1900 {
				append using "`temp'"
				erase "`temp'"
			}
			tempfile temp
			save "`temp'", replace
			clear
		}
		
		forvalues year = 1915/2013 {
			import delimited StataFiles/SimulationMatlab/output/indiv_bysheet/gperc_indiv_`year'.csv
			gen year=`year'
			append using "`temp'"
			erase "`temp'"
			tempfile temp
			save "`temp'", replace
			clear
		}
	* From 2014, use interpolations in a different format
		import delimited StataFiles/SimulationMatlab/output/indiv_bysheet/gperc_indiv_2014.csv, clear
		replace year=2014
		replace b ="." if b=="Inf"
		destring b, replace
		rename b b_indiv
		rename thr thr_indiv
		rename topav ytop_indiv
		keep year p b_indiv thr_indiv ytop_indiv
	* Merge
		append using "`temp'"
		replace p=round(p*1000) if year <2014
		replace p=round(p*100000) if year ==2014
		erase "`temp'"
		sort year p
		save StataFiles/temp/gperc_indiv_1900_2014.dta, replace
			

/******************importing gperc tax unit series for taxable income 1900-2014*******************/
	*1900-2013
		clear
		foreach year in 1900 1910 {
			import delimited StataFiles/SimulationMatlab/output/bysheet/gperc_`year'.csv
			gen year=`year'
			if `year'>1900 {
				append using "`temp'"
				erase "`temp'"
			}
			tempfile temp
			save "`temp'", replace
			clear
		}
		forvalues year = 1915/2013 {
			import delimited StataFiles/SimulationMatlab/output/bysheet/gperc_`year'.csv
			gen year=`year'
			append using "`temp'"
			erase "`temp'"
			tempfile temp
			save "`temp'", replace
			clear
		}

	* From 2014, use interpolations in a different format
		import delimited StataFiles/SimulationMatlab/output/bysheet/gperc_2014.csv
		replace year = 2014
		replace b ="." if b=="Inf"
		destring b, replace
		rename topavg ytop
		keep year p thr ytop b
	
	* Merge
		append using "`temp'"
		erase "`temp'"
		replace p=round(p*1000) if year <2014
		replace p=round(p*100000) if year ==2014
		sort year p
		merge 1:1 year p using StataFiles/temp/gperc_indiv_1900_2014.dta
		drop _merge
		*** Correction from 2014 onwards
			replace thr_unit = thr if year==2014
			replace ytop_unit = ytop if year==2014
			replace b_unit = b if year==2014
		save StataFiles/gperc_fiscalincome_1900_2014.dta, replace


/*******************from taxable income to fiscal income*********************/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	merge m:1 year using StataFiles/temp/averages.dta, nogen 
	rename thr ythr
	rename thr_unit ythrtu
	rename ytop_unit ytoptu
	rename thr_indiv ythrad
	rename ytop_indiv ytopad
	sort year p
	gen yint=ytop if p==99999
	gen yinttu=ytoptu if p==99999
	gen yintad=ytopad if p==99999
	replace yint=((100000-p)*ytop-(100000-p[_n+1])*ytop[_n+1])/(p[_n+1]-p) if p<99999
	replace yinttu=((100000-p)*ytoptu-(100000-p[_n+1])*ytoptu[_n+1])/(p[_n+1]-p) if p<99999
	replace yintad=((100000-p)*ytopad-(100000-p[_n+1])*ytopad[_n+1])/(p[_n+1]-p) if p<99999
	gen ythrt=ythr
	gen ytopt=ytop
	gen yintt=yint
	* Only 10 problematic observations in 1943 for yint
		replace yint=(ythr+ythr[_n+1])/2 if year==1943 & (yint<ythr | yint>ythr[_n+1]) & year==year[_n+1]	
	/*check income ordering of simulated g-percentiles
		sort year p
		gen x=0
		replace x=1 if p>0 & yint<yint[_n-1]
		ta year if x==1
		gen y=0
		replace y=1 if p<99999 & (yint<ythr | yint>ythr[_n+1])
		ta year if y==1
		gen xtu=0
		replace xtu=1 if p>0 & yinttu<yinttu[_n-1]
		ta year if xtu==1
		gen ytu=0
		replace ytu=1 if p<99999 & (yinttu<ythrtu | yinttu>ythrtu[_n+1])
		ta year if ytu==1
		gen xad=0
		replace xad=1 if p>0 & yintad<yintad[_n-1]
		ta year if xad==1
		gen yad=0
		replace yad=1 if p<99999 & (yintad<ythrad | yintad>ythrad[_n+1])
		ta year if yad==1
		drop x-yad
*/
*************************correction for previous-year-tax-deduction************/
	*smoothing correction factors
		sort year p
		gen corrt=corrt00 if p<=85000
		replace corrt=corrt[_n-1]+(corrt90-corrt00)/5 if p>85000 & p<=90000
		replace corrt=corrt[_n-1]+(corrt95-corrt90)/5 if p>90000 & p<=95000
		replace corrt=corrt[_n-1]+(corrt99-corrt95)/4 if p>95000 & p<=99000
		replace corrt=corrt[_n-1]+(corrt995-corrt99)/5 if p>99000 & p<=99500
		replace corrt=corrt[_n-1]+(corrt999-corrt995)/4 if p>99500 & p<=99900
		replace corrt=corrt[_n-1]+(corrt9999-corrt999)/9 if p>99900 & p<=99990
		replace corrt=corrt9999 if p>99990
	* apply correction factors
		replace yint=corrt*yint
		replace ythr=corrt*ythr
		replace yinttu=corrt*yinttu
		replace ythrtu=corrt*ythrtu
		replace yintad=corrt*yintad
		replace ythrad=corrt*ythrad
	* 2 problematic observations in 1965
		replace yint=(ythr+ythr[_n+1])/2 if year==1965 & (yint<ythr | yint>ythr[_n+1]) & year==year[_n+1]
		replace yinttu=(ythrtu+ythrtu[_n+1])/2 if year==1965 & (yinttu<ythrtu | yinttu>ythrtu[_n+1]) & year==year[_n+1]
	/*check income ordering of simulated g-percentiles
		sort year p
		gen x=0
		replace x=1 if p>0 & yint<yint[_n-1]
		ta year if x==1
		gen y=0
		replace y=1 if p<99999 & (yint<ythr | yint>ythr[_n+1])
		ta year if y==1
		gen xtu=0
		replace xtu=1 if p>0 & yinttu<yinttu[_n-1]
		ta year if xtu==1
		gen ytu=0
		replace ytu=1 if p<99999 & (yinttu<ythrtu | yinttu>ythrtu[_n+1])
		ta year if ytu==1
		gen xad=0
		replace xad=1 if p>0 & yintad<yintad[_n-1]
		ta year if xad==1
		gen yad=0
		replace yad=1 if p<99999 & (yintad<ythrad | yintad>ythrad[_n+1])
		ta year if yad==1
		drop x-yad
	*/
		
	keep year p ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ayttu ayftu* ayntu aytad ayfad* aynad corr*
	order year p ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ayttu ayftu ayntu aytad ayfad aynad corr*
	sort year p ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ayttu ayftu ayntu aytad ayfad aynad corr*

	* Adjust average taxable incomes
		gen f=1 if p==99999
		replace f=p[_n+1]-p if p<99999
		by year: egen ay=total(f*yint)
		by year: egen aytu=total(f*yinttu)
		by year: egen ayad=total(f*yintad)
		replace ay=ay/100000
		replace aytu=aytu/100000
		replace ayad=ayad/100000
		gen ratio=ay/ayttu
		gen ratiotu=aytu/ayttu
		gen ratioad=ayad/aytad
	/****************check averages: ratios should be very close to one************/
	/***check done 15-5-16: all ratios in 0.999-1.001 interval****/
	/*collapse ay aytu ayttu ayad aytad ratio ratiotu ratioad, by (year)*/
	/***************************adjust average taxable incomes***********************/
		replace ythr=ythr/ratio
		replace yint=yint/ratio
		replace ythrtu=ythrtu/ratiotu
		replace yinttu=yinttu/ratiotu
		replace ythrad=ythrad/ratioad
		replace yintad=yintad/ratioad
keep year p f ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ayttu ayftu* ayntu aytad ayfad* aynad corr*
order year p f ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ayttu ayftu ayntu aytad ayfad aynad corr*
sort year p f ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ayttu ayftu ayntu aytad ayfad aynad corr*
save StataFiles/gperc_fiscalincome_1900_2014.dta, replace


/*************correction for other deductions (including standard 10%-20% deduction for professional expenses)************/
	*smoothing correction factors
		gen corr=corr00 if p<=85000
		replace corr=corr[_n-1]+(corr90-corr00)/5 if p>85000 & p<=90000
		replace corr=corr[_n-1]+(corr95-corr90)/5 if p>90000 & p<=95000
		replace corr=corr[_n-1]+(corr99-corr95)/4 if p>95000 & p<=99000
		replace corr=corr[_n-1]+(corr995-corr99)/5 if p>99000 & p<=99500
		replace corr=corr[_n-1]+(corr999-corr995)/4 if p>99500 & p<=99900
		replace corr=corr[_n-1]+(corr9999-corr999)/9 if p>99900 & p<=99990
		replace corr=corr9999 if p>99990
	* apply correction factors
		replace yint=corr*yint
		replace ythr=corr*ythr
		replace yinttu=corr*yinttu
		replace ythrtu=corr*ythrtu
		replace yintad=corr*yintad
		replace ythrad=corr*ythrad
	* 2 problematic observations in 1965
		sort year p
		replace yint=(ythr+ythr[_n+1])/2 if year==1965 & (yint<ythr | yint>ythr[_n+1]) & year==year[_n+1]
		replace yinttu=(ythrtu+ythrtu[_n+1])/2 if year==1965 & (yinttu<ythrtu | yinttu>ythrtu[_n+1]) & year==year[_n+1]
	/*check income ordering of simulated g-percentiles
		sort year p
		gen x=0
		replace x=1 if p>0 & yint<yint[_n-1]
		ta year if x==1
		gen y=0
		replace y=1 if p<99999 & (yint<ythr | yint>ythr[_n+1])
		ta year if y==1
		gen xtu=0
		replace xtu=1 if p>0 & yinttu<yinttu[_n-1]
		ta year if xtu==1
		gen ytu=0
		replace ytu=1 if p<99999 & (yinttu<ythrtu | yinttu>ythrtu[_n+1])
		ta year if ytu==1
		gen xad=0
		replace xad=1 if p>0 & yintad<yintad[_n-1]
		ta year if xad==1
		gen yad=0
		replace yad=1 if p<99999 & (yintad<ythrad | yintad>ythrad[_n+1])
		ta year if yad==1
		drop x-yad
	*/
	
	save StataFiles/gperc_fiscalincome_1900_2014.dta, replace

/******************************correction for capital gains************/
	*A) start with years 1900-1987 and 1989-2002 (tabulations exclude capital gains)
			*smoothing correction factors
				gen corrcg=corr00cg if p<=85000
				replace corrcg=corrcg[_n-1]+(corr90cg-corr00cg)/5 if p>85000 & p<=90000
				replace corrcg=corrcg[_n-1]+(corr95cg-corr90cg)/5 if p>90000 & p<=95000
				replace corrcg=corrcg[_n-1]+(corr99cg-corr95cg)/4 if p>95000 & p<=99000
				replace corrcg=corrcg[_n-1]+(corr995cg-corr99cg)/5 if p>99000 & p<=99500
				replace corrcg=corrcg[_n-1]+(corr999cg-corr995cg)/4 if p>99500 & p<=99900
				replace corrcg=corrcg[_n-1]+(corr9999cg-corr999cg)/9 if p>99900 & p<=99990
				replace corrcg=corr9999cg if p>99990
				replace corrcg=corr9999cg if p>=99999
			* Apply correction factors
				drop if year==1988 | year>=2003
				gen yintcg=corrcg*yint
				gen ythrcg=corrcg*ythr
				gen yinttucg=corrcg*yinttu
				gen ythrtucg=corrcg*ythrtu
				gen yintadcg=corrcg*yintad
				gen ythradcg=corrcg*ythrad
			/*check income ordering of simulated g-percentiles
				* with cg
					sort year p
					gen x=0
					replace x=1 if p>0 & yintcg<yintcg[_n-1]
					ta year if x==1
					gen y=0
					replace y=1 if p<99999 & (yintcg<ythrcg | yintcg>ythrcg[_n+1])
					ta year if y==1
					gen xtu=0
					replace xtu=1 if p>0 & yinttucg<yinttucg[_n-1]
					ta year if xtu==1
					gen ytu=0
					replace ytu=1 if p<99999 & (yinttucg<ythrtucg | yinttucg>ythrtucg[_n+1])
					ta year if ytu==1
					gen xad=0
					replace xad=1 if p>0 & yintadcg<yintadcg[_n-1]
					ta year if xad==1
					gen yad=0
					replace yad=1 if p<99999 & (yintadcg<ythradcg | yintadcg>ythradcg[_n+1])
					ta year if yad==1
					drop x-yad
			*/
			save StataFiles/temp/gperc_1900_2014_yncg.dta, replace
	* B) continue with years 1988 and 2003-2014 (tabulations include capital gains)

		* Compute correction factors for capital gains 2003-2014: tabulations include capital gains since 2003 Table D4 R161
			
				cd "$projdir/GGP2017DINAAppendixC/"
				use DataFiles/Results/gperc/gpercr_irpp_foy_19702014.dta,clear
				foreach var of varlist yint {
					rename `var' `var'_ecg
				}
				keep if year >=2003 & sex==0
				* Percentile
					gen p =gperc
					replace p=p*10 if gperc<99990
					replace p=p*10 if gperc<9990
					replace p=p*10 if gperc<990				
					keep year p yint_ecg
				* Merge with tabulation including capital gains
					cd "$projdir/GGP2017DINAAppendixD/"
					merge 1:1 year p using StataFiles/gperc_fiscalincome_1900_2014.dta
					keep if year==1988 | year>=2003
					drop _merge
				* % capital gains by income groups	
					gen corrcgr=yinttu/yint_ecg if p>=90000
				
		*smoothing correction factors for year 1988 or bottom 90
			sort year p
			replace corrcgr=corr00cgr if p<=85000
			replace corrcgr=corrcgr[_n-1]+(corr90cgr-corr00cgr)/5 if p>85000 & p<90000
			replace corrcgr=corrcgr[_n-1]+(corr95cgr-corr90cgr)/5 if p>=90000 & p<=95000 & year==1988
			replace corrcgr=corrcgr[_n-1]+(corr99cgr-corr95cgr)/4 if p>95000 & p<=99000  & year==1988
			replace corrcgr=corrcgr[_n-1]+(corr995cgr-corr99cgr)/5 if p>99000 & p<=99500  & year==1988
			replace corrcgr=corrcgr[_n-1]+(corr999cgr-corr995cgr)/4 if p>99500 & p<=99900  & year==1988
			replace corrcgr=corrcgr[_n-1]+(corr9999cgr-corr999cgr)/9 if p>99900 & p<=99990 & year==1988
			replace corrcgr=corr9999cgr if p>99990 & year==1988
		* apply correction factors																			
			replace yint=yint/corrcgr
			replace ythr=ythr/corrcgr
			replace yinttu=yinttu/corrcgr
			replace ythrtu=ythrtu/corrcgr
			replace yintad=yintad/corrcgr
			replace ythrad=ythrad/corrcgr
			gen yintcg=corrcg*yint
			gen ythrcg=corrcg*ythr
			gen yinttucg=corrcg*yinttu
			gen ythrtucg=corrcg*ythrtu
			gen yintadcg=corrcg*yintad
			gen ythradcg=corrcg*ythrad
		* 3 problematic observations in 2010
			sort year p
			replace yint=(ythr+ythr[_n+1])/2 if year==2010 & (yint<ythr | yint>ythr[_n+1]) & year==year[_n+1]
			replace yinttu=(ythrtu+ythrtu[_n+1])/2 if year==2010 & (yinttu<ythrtu | yinttu>ythrtu[_n+1]) & year==year[_n+1]
			replace yintad=(ythrad+ythrad[_n+1])/2 if year==2010 & (yintad<ythrad | yintad>ythrad[_n+1]) & year==year[_n+1]
			
		save StataFiles/temp/gperc_1900_2014_ycg.dta, replace
		append using StataFiles/temp/gperc_1900_2014_yncg.dta
		save StataFiles/gperc_fiscalincome_1900_2014.dta, replace
		/*check income ordering of simulated g-percentiles
			* with cg
				sort year p
				gen x=0
				replace x=1 if p>0 & yintcg<yintcg[_n-1]
				ta year if x==1
				gen y=0
				replace y=1 if p<99999 & (yintcg<ythrcg | yintcg>ythrcg[_n+1])
				ta year if y==1
				gen xtu=0
				replace xtu=1 if p>0 & yinttucg<yinttucg[_n-1]
				ta year if xtu==1
				gen ytu=0
				replace ytu=1 if p<99999 & (yinttucg<ythrtucg | yinttucg>ythrtucg[_n+1])
				ta year if ytu==1
				gen xad=0
				replace xad=1 if p>0 & yintadcg<yintadcg[_n-1]
				ta year if xad==1
				gen yad=0
				replace yad=1 if p<99999 & (yintadcg<ythradcg | yintadcg>ythradcg[_n+1])
				ta year if yad==1
				drop x-yad
			* without cg
				sort year p
				gen x=0
				replace x=1 if p>0 & yint<yint[_n-1]
				ta year if x==1
				gen y=0
				replace y=1 if p<99999 & (yint<ythr | yint>ythr[_n+1])
				ta year if y==1
				gen xtu=0
				replace xtu=1 if p>0 & yinttu<yinttu[_n-1]
				ta year if xtu==1
				gen ytu=0
				replace ytu=1 if p<99999 & (yinttu<ythrtu | yinttu>ythrtu[_n+1])
				ta year if ytu==1
				gen xad=0
				replace xad=1 if p>0 & yintad<yintad[_n-1]
				ta year if xad==1
				gen yad=0
				replace yad=1 if p<99999 & (yintad<ythrad | yintad>ythrad[_n+1])
				ta year if yad==1
				drop x-yad	
		*/

/**********************check and adjust average fiscal incomes***********/
use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
sort year p
by year: egen ay=total(f*yint)
by year: egen aytu=total(f*yinttu)
by year: egen ayad=total(f*yintad)
by year: egen aycg=total(f*yintcg)
by year: egen aytucg=total(f*yinttucg)
by year: egen ayadcg=total(f*yintadcg)
replace ay=ay/100000
replace aytu=aytu/100000
replace ayad=ayad/100000
replace aycg=aycg/100000
replace aytucg=aytucg/100000
replace ayadcg=ayadcg/100000
gen ratio=ay/ayftu
gen ratiotu=aytu/ayftu
gen ratioad=ayad/ayfad
gen ratiocg=aycg/ayftucg
gen ratiotucg=aytucg/ayftucg
gen ratioadcg=ayadcg/ayfadcg
/*************check averages: ratios should be very close to one*/
/***check done 15-5-16: all ratios in 0.999-1.001 interval****/
/*collapse ay aytu ayftu ayad ayfad ratio*, by (year)*/


/******************adjust average fiscal incomes*/
replace ythr=ythr/ratio
replace yint=yint/ratio
replace ythrtu=ythrtu/ratiotu
replace yinttu=yinttu/ratiotu
replace ythrad=ythrad/ratioad
replace yintad=yintad/ratioad
replace ythrcg=ythrcg/ratiocg
replace yintcg=yintcg/ratiocg
replace ythrtucg=ythrtucg/ratiotucg
replace yinttucg=yinttucg/ratiotucg
replace ythradcg=ythradcg/ratioadcg
replace yintadcg=yintadcg/ratioadcg
keep year p f ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ythrcg yintcg ythrtucg yinttucg ythradcg yintadcg
order year p f ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ythrcg yintcg ythrtucg yinttucg ythradcg yintadcg
sort year p f ythrt yintt ytopt ythr yint ythrtu yinttu ythrad yintad ythrcg yintcg ythrtucg yinttucg ythradcg yintadcg
save StataFiles/gperc_fiscalincome_1900_2014.dta, replace


/*******computing ytop and b**************************************/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	gsort year -p
	gen ytop=yint if p==99999
	replace ytop=(f*yint+(100000-p[_n-1])*ytop[_n-1])/(100000-p) if p<99999
	gen b=ytop/ythr
	gen ytoptu=yinttu if p==99999
	replace ytoptu=(f*yinttu+(100000-p[_n-1])*ytoptu[_n-1])/(100000-p) if p<99999
	gen btu=ytoptu/ythrtu
	gen ytopad=yintad if p==99999
	replace ytopad=(f*yintad+(100000-p[_n-1])*ytopad[_n-1])/(100000-p) if p<99999
	gen bad=ytopad/ythrad
	gen ytopcg=yintcg if p==99999
	replace ytopcg=(f*yintcg+(100000-p[_n-1])*ytopcg[_n-1])/(100000-p) if p<99999
	gen bcg=ytopcg/ythrcg
	gen ytoptucg=yinttucg if p==99999
	replace ytoptucg=(f*yinttucg+(100000-p[_n-1])*ytoptucg[_n-1])/(100000-p) if p<99999
	gen btucg=ytoptucg/ythrtucg
	gen ytopadcg=yintadcg if p==99999
	replace ytopadcg=(f*yintadcg+(100000-p[_n-1])*ytopadcg[_n-1])/(100000-p) if p<99999
	gen badcg=ytopadcg/ythradcg
	keep year p f ythrt yintt ytopt ythr yint ytop b ythrtu yinttu ytoptu btu ythrad yintad ytopad bad ythrcg yintcg ytopcg bcg ythrtucg yinttucg ytoptucg btucg ythradcg yintadcg ytopadcg badcg
	order year p f ythrt yintt ytopt ythr yint ytop b ythrtu yinttu ytoptu btu ythrad yintad ytopad bad ythrcg yintcg ytopcg bcg ythrtucg yinttucg ytoptucg btucg ythradcg yintadcg ytopadcg badcg
	sort year p f ythrt yintt ytopt ythr yint ytop b ythrtu yinttu ytoptu btu ythrad yintad ytopad bad ythrcg yintcg ytopcg bcg ythrtucg yinttucg ytoptucg btucg ythradcg yintadcg ytopadcg badcg
	save StataFiles/gperc_fiscalincome_1900_2014.dta, replace


/*extracting table D10: summary statistics for distribution of fiscal income (excl. capital gains) among tax units*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	keep year p f ythr yint ytop
	sort year p
	gen p0=0 if p==0
	replace p0=p/1000+(1-yint/(ythr[_n+1]-ythr))*f/100000 if ythr==0 & ythr[_n+1]>0 & year[_n+1]==year
	keep if p==0 | p==10000 | p==50000 | p==90000 | p==99000 | p==99900 | p==99990 | p==99999
	sort year p 
	gen ymean=ytop if p==0
	gen bottom10=1-0.9*ytop[_n+1]/ytop if p==0
	gen bottom50=1-0.5*ytop[_n+2]/ytop if p==0
	gen top10=0.1*ytop[_n+3]/ytop if p==0
	gen middle40=1-top10-bottom50 if p==0
	gen top1=0.01*ytop[_n+4]/ytop if p==0
	gen top01=0.001*ytop[_n+5]/ytop if p==0
	gen top001=0.0001*ytop[_n+6]/ytop if p==0
	gen top0001=0.00001*ytop[_n+7]/ytop if p==0
	gen p10=ythr[_n+1]/ymean if p==0
	gen p50=ythr[_n+2]/ymean if p==0
	gen p90=ythr[_n+3]/ymean if p==0
	gen p99=ythr[_n+4]/ymean if p==0
	gen p999=ythr[_n+5]/ymean if p==0
	gen p9999=ythr[_n+6]/ymean if p==0
	gen p99999=ythr[_n+7]/ymean if p==0
	gen b10=ytop[_n+1]/ythr[_n+1] if p==0
	gen b50=ytop[_n+2]/ythr[_n+2] if p==0
	gen b90=ytop[_n+3]/ythr[_n+3] if p==0
	gen b99=ytop[_n+4]/ythr[_n+4] if p==0
	gen b999=ytop[_n+5]/ythr[_n+5] if p==0
	gen b9999=ytop[_n+6]/ythr[_n+6] if p==0
	gen b99999=ytop[_n+7]/ythr[_n+7] if p==0
	keep if p==0
	keep year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	order year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	save StataFiles/tables/TableD10.dta, replace
	export excel using StataFiles/tables/TableD10.xlsx, firstrow(variables) replace

/*************************extracting table D10a: summary statistics for distribution of taxable income (after all deductions, incl. previous-income-ax, i.e. taxable income as defined in tabulations) among tax units*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	keep year p f ytopt yintt ythrt
	rename ytopt ytop
	rename ythrt ythr
	rename yintt yint
	sort year p
	gen p0=0 if p==0
	replace p0=p/1000+(1-yint/(ythr[_n+1]-ythr))*f/100000 if ythr==0 & ythr[_n+1]>0 & year[_n+1]==year
	keep if p==0 | p==10000 | p==50000 | p==90000 | p==99000 | p==99900 | p==99990 | p==99999
	sort year p 
	gen ymean=ytop if p==0
	gen bottom10=1-0.9*ytop[_n+1]/ytop if p==0
	gen bottom50=1-0.5*ytop[_n+2]/ytop if p==0
	gen top10=0.1*ytop[_n+3]/ytop if p==0
	gen middle40=1-top10-bottom50 if p==0
	gen top1=0.01*ytop[_n+4]/ytop if p==0
	gen top01=0.001*ytop[_n+5]/ytop if p==0
	gen top001=0.0001*ytop[_n+6]/ytop if p==0
	gen top0001=0.00001*ytop[_n+7]/ytop if p==0
	gen p10=ythr[_n+1]/ymean if p==0
	gen p50=ythr[_n+2]/ymean if p==0
	gen p90=ythr[_n+3]/ymean if p==0
	gen p99=ythr[_n+4]/ymean if p==0
	gen p999=ythr[_n+5]/ymean if p==0
	gen p9999=ythr[_n+6]/ymean if p==0
	gen p99999=ythr[_n+7]/ymean if p==0
	gen b10=ytop[_n+1]/ythr[_n+1] if p==0
	gen b50=ytop[_n+2]/ythr[_n+2] if p==0
	gen b90=ytop[_n+3]/ythr[_n+3] if p==0
	gen b99=ytop[_n+4]/ythr[_n+4] if p==0
	gen b999=ytop[_n+5]/ythr[_n+5] if p==0
	gen b9999=ytop[_n+6]/ythr[_n+6] if p==0
	gen b99999=ytop[_n+7]/ythr[_n+7] if p==0
	keep if p==0
	keep year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	order year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	save StataFiles/temp/TableD10a.dta, replace
	export excel using StataFiles/temp/exportresults_DINA_ApD.xlsx, sheet("TD10a") sheetmodify firstrow(variables)

/*************************extracting table D10b: summary statistics for distribution of fiscal income among tax units (estimated from equal-split individuals)*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	keep year p f ytoptu yinttu ythrtu
	rename ytoptu ytop
	rename ythrtu ythr
	rename yinttu yint
	sort year p
	gen p0=0 if p==0
	replace p0=p/1000+(1-yint/(ythr[_n+1]-ythr))*f/100000 if ythr==0 & ythr[_n+1]>0 & year[_n+1]==year
	keep if p==0 | p==10000 | p==50000 | p==90000 | p==99000 | p==99900 | p==99990 | p==99999
	sort year p 
	gen ymean=ytop if p==0
	gen bottom10=1-0.9*ytop[_n+1]/ytop if p==0
	gen bottom50=1-0.5*ytop[_n+2]/ytop if p==0
	gen top10=0.1*ytop[_n+3]/ytop if p==0
	gen middle40=1-top10-bottom50 if p==0
	gen top1=0.01*ytop[_n+4]/ytop if p==0
	gen top01=0.001*ytop[_n+5]/ytop if p==0
	gen top001=0.0001*ytop[_n+6]/ytop if p==0
	gen top0001=0.00001*ytop[_n+7]/ytop if p==0
	gen p10=ythr[_n+1]/ymean if p==0
	gen p50=ythr[_n+2]/ymean if p==0
	gen p90=ythr[_n+3]/ymean if p==0
	gen p99=ythr[_n+4]/ymean if p==0
	gen p999=ythr[_n+5]/ymean if p==0
	gen p9999=ythr[_n+6]/ymean if p==0
	gen p99999=ythr[_n+7]/ymean if p==0
	gen b10=ytop[_n+1]/ythr[_n+1] if p==0
	gen b50=ytop[_n+2]/ythr[_n+2] if p==0
	gen b90=ytop[_n+3]/ythr[_n+3] if p==0
	gen b99=ytop[_n+4]/ythr[_n+4] if p==0
	gen b999=ytop[_n+5]/ythr[_n+5] if p==0
	gen b9999=ytop[_n+6]/ythr[_n+6] if p==0
	gen b99999=ytop[_n+7]/ythr[_n+7] if p==0
	keep if p==0
	keep year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	order year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	save StataFiles/temp/TableD10b.dta, replace
	export excel using StataFiles/temp/exportresults_DINA_ApD.xlsx, sheet("TD10b") sheetmodify firstrow(variables)

/*******************extracting table D11: detailed statistics for distribution of fiscal income (excl. capital gains) among tax units*/
	foreach year in 1900 1910 {
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	forvalues year=1915/2014 {
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	use StataFiles/temp/g_percf_1900.dta
	merge 1:1 p using StataFiles/temp/g_percf_1910.dta, nogen
	forvalues year=1915/2014 {
		merge 1:1 p using StataFiles/temp/g_percf_`year'.dta, nogen
	}
	gen b_1900_10=(b_1900+b_1910)/2
	gen b_1915_19=(b_1915+b_1916+b_1917+b_1918+b_1919)/5
	gen b_1920_29=(b_1920+b_1921+b_1922+b_1923+b_1924+b_1925+b_1926+b_1927+b_1928+b_1929)/10
	gen b_1930_39=(b_1930+b_1931+b_1932+b_1933+b_1934+b_1935+b_1936+b_1937+b_1938+b_1939)/10
	gen b_1940_49=(b_1940+b_1941+b_1942+b_1943+b_1944+b_1945+b_1946+b_1947+b_1948+b_1949)/10
	gen b_1950_59=(b_1950+b_1951+b_1952+b_1953+b_1954+b_1955+b_1956+b_1957+b_1958+b_1959)/10
	gen b_1960_69=(b_1960+b_1961+b_1962+b_1963+b_1964+b_1965+b_1966+b_1967+b_1968+b_1969)/10
	gen b_1970_79=(b_1970+b_1971+b_1972+b_1973+b_1974+b_1975+b_1976+b_1977+b_1978+b_1979)/10
	gen b_1980_89=(b_1980+b_1981+b_1982+b_1983+b_1984+b_1985+b_1986+b_1987+b_1988+b_1989)/10
	gen b_1990_99=(b_1990+b_1991+b_1992+b_1993+b_1994+b_1995+b_1996+b_1997+b_1998+b_1999)/10
	gen b_2000_09=(b_2000+b_2001+b_2002+b_2003+b_2004+b_2005+b_2006+b_2007+b_2008+b_2009)/10
	gen b_2010_13=(b_2010+b_2011+b_2012+b_2013)/4
	save StataFiles/tables/TableD11.dta, replace
	export excel using StataFiles/tables/TableD11.xlsx, firstrow(variables) replace


/*******************extracting table D11a: detailed statistics for distribution of taxable income among tax units*/
	foreach year in 1900 1910 {
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythrt
		replace ytop=ytopt
		replace b=ytop/ythr
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	forvalues year=1915/2014 {
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythrt
		replace ytop=ytopt
		replace b=ytop/ythr
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	use StataFiles/temp/g_percf_1900.dta
	merge 1:1 p using StataFiles/temp/g_percf_1910.dta, nogen
	forvalues year=1915/2014 {
	merge 1:1 p using StataFiles/temp/g_percf_`year'.dta, nogen
	}
	gen b_1900_10=(b_1900+b_1910)/2
	gen b_1915_19=(b_1915+b_1916+b_1917+b_1918+b_1919)/5
	gen b_1920_29=(b_1920+b_1921+b_1922+b_1923+b_1924+b_1925+b_1926+b_1927+b_1928+b_1929)/10
	gen b_1930_39=(b_1930+b_1931+b_1932+b_1933+b_1934+b_1935+b_1936+b_1937+b_1938+b_1939)/10
	gen b_1940_49=(b_1940+b_1941+b_1942+b_1943+b_1944+b_1945+b_1946+b_1947+b_1948+b_1949)/10
	gen b_1950_59=(b_1950+b_1951+b_1952+b_1953+b_1954+b_1955+b_1956+b_1957+b_1958+b_1959)/10
	gen b_1960_69=(b_1960+b_1961+b_1962+b_1963+b_1964+b_1965+b_1966+b_1967+b_1968+b_1969)/10
	gen b_1970_79=(b_1970+b_1971+b_1972+b_1973+b_1974+b_1975+b_1976+b_1977+b_1978+b_1979)/10
	gen b_1980_89=(b_1980+b_1981+b_1982+b_1983+b_1984+b_1985+b_1986+b_1987+b_1988+b_1989)/10
	gen b_1990_99=(b_1990+b_1991+b_1992+b_1993+b_1994+b_1995+b_1996+b_1997+b_1998+b_1999)/10
	gen b_2000_09=(b_2000+b_2001+b_2002+b_2003+b_2004+b_2005+b_2006+b_2007+b_2008+b_2009)/10
	gen b_2010_13=(b_2010+b_2011+b_2012+b_2013)/4
	save StataFiles/temp/TableD11a.dta, replace
	export excel using StataFiles/temp/exportresults_DINA_ApD.xlsx, sheet("TD11a") sheetmodify firstrow(variables)

/********************extracting table D12: summary statistics for distribution of fiscal income incl. capital gains among tax unis*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	keep year p f yintcg ytopcg ythrcg
	rename ytopcg ytop
	rename ythrcg ythr
	rename yintcg yint
	sort year p
	gen p0=0 if p==0
	replace p0=p/1000+(1-yint/(ythr[_n+1]-ythr))*f/100000 if ythr==0 & ythr[_n+1]>0 & year[_n+1]==year
	keep if p==0 | p==10000 | p==50000 | p==90000 | p==99000 | p==99900 | p==99990 | p==99999
	sort year p 
	gen ymean=ytop if p==0
	gen bottom10=1-0.9*ytop[_n+1]/ytop if p==0
	gen bottom50=1-0.5*ytop[_n+2]/ytop if p==0
	gen top10=0.1*ytop[_n+3]/ytop if p==0
	gen middle40=1-top10-bottom50 if p==0
	gen top1=0.01*ytop[_n+4]/ytop if p==0
	gen top01=0.001*ytop[_n+5]/ytop if p==0
	gen top001=0.0001*ytop[_n+6]/ytop if p==0
	gen top0001=0.00001*ytop[_n+7]/ytop if p==0
	gen p10=ythr[_n+1]/ymean if p==0
	gen p50=ythr[_n+2]/ymean if p==0
	gen p90=ythr[_n+3]/ymean if p==0
	gen p99=ythr[_n+4]/ymean if p==0
	gen p999=ythr[_n+5]/ymean if p==0
	gen p9999=ythr[_n+6]/ymean if p==0
	gen p99999=ythr[_n+7]/ymean if p==0
	gen b10=ytop[_n+1]/ythr[_n+1] if p==0
	gen b50=ytop[_n+2]/ythr[_n+2] if p==0
	gen b90=ytop[_n+3]/ythr[_n+3] if p==0
	gen b99=ytop[_n+4]/ythr[_n+4] if p==0
	gen b999=ytop[_n+5]/ythr[_n+5] if p==0
	gen b9999=ytop[_n+6]/ythr[_n+6] if p==0
	gen b99999=ytop[_n+7]/ythr[_n+7] if p==0
	keep if p==0
	keep year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	order year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	save StataFiles/tables/TableD12.dta, replace
	export excel using StataFiles/tables/TableD12.xlsx, firstrow(variables) replace

/*******************extracting table D13: detailed statistics for distribution of fiscal income incl.cap.gains among tax units*/
	foreach year in 1900 1910{
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythrcg
		replace ytop=ytopcg
		replace b=ytopcg/ythrcg
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	forvalues year=1915/2014{
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythrcg
		replace ytop=ytopcg
		replace b=ytopcg/ythrcg
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	use StataFiles/temp/g_percf_1900.dta
	merge 1:1 p using StataFiles/temp/g_percf_1910.dta, nogen
	forvalues year=1915/2014{
		merge 1:1 p using StataFiles/temp/g_percf_`year'.dta, nogen
	}
	gen b_1900_10=(b_1900+b_1910)/2
	gen b_1915_19=(b_1915+b_1916+b_1917+b_1918+b_1919)/5
	gen b_1920_29=(b_1920+b_1921+b_1922+b_1923+b_1924+b_1925+b_1926+b_1927+b_1928+b_1929)/10
	gen b_1930_39=(b_1930+b_1931+b_1932+b_1933+b_1934+b_1935+b_1936+b_1937+b_1938+b_1939)/10
	gen b_1940_49=(b_1940+b_1941+b_1942+b_1943+b_1944+b_1945+b_1946+b_1947+b_1948+b_1949)/10
	gen b_1950_59=(b_1950+b_1951+b_1952+b_1953+b_1954+b_1955+b_1956+b_1957+b_1958+b_1959)/10
	gen b_1960_69=(b_1960+b_1961+b_1962+b_1963+b_1964+b_1965+b_1966+b_1967+b_1968+b_1969)/10
	gen b_1970_79=(b_1970+b_1971+b_1972+b_1973+b_1974+b_1975+b_1976+b_1977+b_1978+b_1979)/10
	gen b_1980_89=(b_1980+b_1981+b_1982+b_1983+b_1984+b_1985+b_1986+b_1987+b_1988+b_1989)/10
	gen b_1990_99=(b_1990+b_1991+b_1992+b_1993+b_1994+b_1995+b_1996+b_1997+b_1998+b_1999)/10
	gen b_2000_09=(b_2000+b_2001+b_2002+b_2003+b_2004+b_2005+b_2006+b_2007+b_2008+b_2009)/10
	gen b_2010_13=(b_2010+b_2011+b_2012+b_2013)/4
	save StataFiles/tables/TableD13.dta, replace
	export excel using StataFiles/tables/TableD13.xlsx, firstrow(variables) replace

/***************extracting table D14: summary statistics for distribution of fiscal income excl. capital gains among equal-split individuals*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	keep year p f yintad ytopad ythrad
	rename ytopad ytop
	rename ythrad ythr
	rename yintad yint
	sort year p
	gen p0=0 if p==0
	replace p0=p/1000+(1-yint/(ythr[_n+1]-ythr))*f/100000 if ythr==0 & ythr[_n+1]>0 & year[_n+1]==year
	keep if p==0 | p==10000 | p==50000 | p==90000 | p==99000 | p==99900 | p==99990 | p==99999
	sort year p 
	gen ymean=ytop if p==0
	gen bottom10=1-0.9*ytop[_n+1]/ytop if p==0
	gen bottom50=1-0.5*ytop[_n+2]/ytop if p==0
	gen top10=0.1*ytop[_n+3]/ytop if p==0
	gen middle40=1-top10-bottom50 if p==0
	gen top1=0.01*ytop[_n+4]/ytop if p==0
	gen top01=0.001*ytop[_n+5]/ytop if p==0
	gen top001=0.0001*ytop[_n+6]/ytop if p==0
	gen top0001=0.00001*ytop[_n+7]/ytop if p==0
	gen p10=ythr[_n+1]/ymean if p==0
	gen p50=ythr[_n+2]/ymean if p==0
	gen p90=ythr[_n+3]/ymean if p==0
	gen p99=ythr[_n+4]/ymean if p==0
	gen p999=ythr[_n+5]/ymean if p==0
	gen p9999=ythr[_n+6]/ymean if p==0
	gen p99999=ythr[_n+7]/ymean if p==0
	gen b10=ytop[_n+1]/ythr[_n+1] if p==0
	gen b50=ytop[_n+2]/ythr[_n+2] if p==0
	gen b90=ytop[_n+3]/ythr[_n+3] if p==0
	gen b99=ytop[_n+4]/ythr[_n+4] if p==0
	gen b999=ytop[_n+5]/ythr[_n+5] if p==0
	gen b9999=ytop[_n+6]/ythr[_n+6] if p==0
	gen b99999=ytop[_n+7]/ythr[_n+7] if p==0
	keep if p==0
	keep year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	order year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	save StataFiles/tables/TableD14.dta, replace
	export excel using StataFiles/tables/TableD14.xlsx, firstrow(variables) replace

/*******************extracting table D15: detailed statistics for distribution of fiscal income excl.cap.gains among equal-split individuals*/
	foreach year in 1900 1910{
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythrad
		replace ytop=ytopad
		replace b=ytopad/ythrad
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	forvalues year=1915/2014{
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythrad
		replace ytop=ytopad
		replace b=ytopad/ythrad
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	use StataFiles/temp/g_percf_1900.dta
	merge 1:1 p using StataFiles/temp/g_percf_1910.dta, nogen
	forvalues year=1915/2014{
		merge 1:1 p using StataFiles/temp/g_percf_`year'.dta, nogen
	}
	gen b_1900_10=(b_1900+b_1910)/2
	gen b_1915_19=(b_1915+b_1916+b_1917+b_1918+b_1919)/5
	gen b_1920_29=(b_1920+b_1921+b_1922+b_1923+b_1924+b_1925+b_1926+b_1927+b_1928+b_1929)/10
	gen b_1930_39=(b_1930+b_1931+b_1932+b_1933+b_1934+b_1935+b_1936+b_1937+b_1938+b_1939)/10
	gen b_1940_49=(b_1940+b_1941+b_1942+b_1943+b_1944+b_1945+b_1946+b_1947+b_1948+b_1949)/10
	gen b_1950_59=(b_1950+b_1951+b_1952+b_1953+b_1954+b_1955+b_1956+b_1957+b_1958+b_1959)/10
	gen b_1960_69=(b_1960+b_1961+b_1962+b_1963+b_1964+b_1965+b_1966+b_1967+b_1968+b_1969)/10
	gen b_1970_79=(b_1970+b_1971+b_1972+b_1973+b_1974+b_1975+b_1976+b_1977+b_1978+b_1979)/10
	gen b_1980_89=(b_1980+b_1981+b_1982+b_1983+b_1984+b_1985+b_1986+b_1987+b_1988+b_1989)/10
	gen b_1990_99=(b_1990+b_1991+b_1992+b_1993+b_1994+b_1995+b_1996+b_1997+b_1998+b_1999)/10
	gen b_2000_09=(b_2000+b_2001+b_2002+b_2003+b_2004+b_2005+b_2006+b_2007+b_2008+b_2009)/10
	gen b_2010_13=(b_2010+b_2011+b_2012+b_2013)/4
	save StataFiles/tables/TableD15.dta, replace
	export excel using StataFiles/tables/TableD15.xlsx, firstrow(variables) replace


/***************extracting table D16: summary statistics for distribution of fiscal income incl. capital gains among equal-split individuals*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	keep year p f yintadcg ytopadcg ythradcg
	rename ytopadcg ytop
	rename ythradcg ythr
	rename yintadcg yint
	sort year p
	gen p0=0 if p==0
	replace p0=p/1000+(1-yint/(ythr[_n+1]-ythr))*f/100000 if ythr==0 & ythr[_n+1]>0 & year[_n+1]==year
	keep if p==0 | p==10000 | p==50000 | p==90000 | p==99000 | p==99900 | p==99990 | p==99999
	sort year p 
	gen ymean=ytop if p==0
	gen bottom10=1-0.9*ytop[_n+1]/ytop if p==0
	gen bottom50=1-0.5*ytop[_n+2]/ytop if p==0
	gen top10=0.1*ytop[_n+3]/ytop if p==0
	gen middle40=1-top10-bottom50 if p==0
	gen top1=0.01*ytop[_n+4]/ytop if p==0
	gen top01=0.001*ytop[_n+5]/ytop if p==0
	gen top001=0.0001*ytop[_n+6]/ytop if p==0
	gen top0001=0.00001*ytop[_n+7]/ytop if p==0
	gen p10=ythr[_n+1]/ymean if p==0
	gen p50=ythr[_n+2]/ymean if p==0
	gen p90=ythr[_n+3]/ymean if p==0
	gen p99=ythr[_n+4]/ymean if p==0
	gen p999=ythr[_n+5]/ymean if p==0
	gen p9999=ythr[_n+6]/ymean if p==0
	gen p99999=ythr[_n+7]/ymean if p==0
	gen b10=ytop[_n+1]/ythr[_n+1] if p==0
	gen b50=ytop[_n+2]/ythr[_n+2] if p==0
	gen b90=ytop[_n+3]/ythr[_n+3] if p==0
	gen b99=ytop[_n+4]/ythr[_n+4] if p==0
	gen b999=ytop[_n+5]/ythr[_n+5] if p==0
	gen b9999=ytop[_n+6]/ythr[_n+6] if p==0
	gen b99999=ytop[_n+7]/ythr[_n+7] if p==0
	keep if p==0
	keep year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	order year ymean bottom10 bottom50 middle40 top10 top1 top01 top001 top0001 p0 p10 p50 p90 p99 p999 p9999 p99999 b10 b50 b90 b99 b999 b9999 b99999
	save StataFiles/tables/TableD16.dta, replace
	export excel using StataFiles/tables/TableD16.xlsx, firstrow(variables) replace

/*******************extracting table D17: detailed statistics for distribution of fiscal income incl.cap.gains among equal-split individuals*/
	foreach year in 1900 1910{
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythradcg
		replace ytop=ytopadcg
		replace b=ytopadcg/ythradcg
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	forvalues year=1915/2014{
		use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
		keep if year==`year'
		replace ythr=ythradcg
		replace ytop=ytopadcg
		replace b=ytopadcg/ythradcg
		rename ythr ythr_`year'
		rename ytop ytop_`year'
		rename b b_`year'
		keep p ythr_`year' ytop_`year' b_`year'
		sort p
		save StataFiles/temp/g_percf_`year'.dta, replace
		clear
	}
	use StataFiles/temp/g_percf_1900.dta
	merge 1:1 p using StataFiles/temp/g_percf_1910.dta, nogen
	forvalues year=1915/2014{
		merge 1:1 p using StataFiles/temp/g_percf_`year'.dta, nogen
	}
	gen b_1900_10=(b_1900+b_1910)/2
	gen b_1915_19=(b_1915+b_1916+b_1917+b_1918+b_1919)/5
	gen b_1920_29=(b_1920+b_1921+b_1922+b_1923+b_1924+b_1925+b_1926+b_1927+b_1928+b_1929)/10
	gen b_1930_39=(b_1930+b_1931+b_1932+b_1933+b_1934+b_1935+b_1936+b_1937+b_1938+b_1939)/10
	gen b_1940_49=(b_1940+b_1941+b_1942+b_1943+b_1944+b_1945+b_1946+b_1947+b_1948+b_1949)/10
	gen b_1950_59=(b_1950+b_1951+b_1952+b_1953+b_1954+b_1955+b_1956+b_1957+b_1958+b_1959)/10
	gen b_1960_69=(b_1960+b_1961+b_1962+b_1963+b_1964+b_1965+b_1966+b_1967+b_1968+b_1969)/10
	gen b_1970_79=(b_1970+b_1971+b_1972+b_1973+b_1974+b_1975+b_1976+b_1977+b_1978+b_1979)/10
	gen b_1980_89=(b_1980+b_1981+b_1982+b_1983+b_1984+b_1985+b_1986+b_1987+b_1988+b_1989)/10
	gen b_1990_99=(b_1990+b_1991+b_1992+b_1993+b_1994+b_1995+b_1996+b_1997+b_1998+b_1999)/10
	gen b_2000_09=(b_2000+b_2001+b_2002+b_2003+b_2004+b_2005+b_2006+b_2007+b_2008+b_2009)/10
	gen b_2010_13=(b_2010+b_2011+b_2012+b_2013)/4
	save StataFiles/tables/TableD17.dta, replace
	export excel using StataFiles/tables/TableD17.xlsx, firstrow(variables) replace


/***************renaming variables as described in codebook*/
	use StataFiles/gperc_fiscalincome_1900_2014.dta, clear
	rename ythr ythr_fninc_t
	rename yint yint_fninc_t
	rename ytop ytop_fninc_t
	rename b b_fninc_t
	rename ythrcg ythr_fiinc_t
	rename yintcg yint_fiinc_t
	rename ytopcg ytop_fiinc_t
	rename bcg b_fiinc_t
	rename ythrad ythr_fninc_j
	rename yintad yint_fninc_j
	rename ytopad ytop_fninc_j
	rename bad b_fninc_j
	rename ythradcg ythr_fiinc_j
	rename yintadcg yint_fiinc_j
	rename ytopadcg ytop_fiinc_j
	rename badcg b_fiinc_j
	keep year p f ythr_fninc_t yint_fninc_t ytop_fninc_t b_fninc_t ythr_fninc_j yint_fninc_j ytop_fninc_j b_fninc_j ythr_fiinc_t yint_fiinc_t ytop_fiinc_t b_fiinc_t ythr_fiinc_j yint_fiinc_j ytop_fiinc_j b_fiinc_j
	order year p f ythr_fninc_t yint_fninc_t ytop_fninc_t b_fninc_t ythr_fninc_j yint_fninc_j ytop_fninc_j b_fninc_j ythr_fiinc_t yint_fiinc_t ytop_fiinc_t b_fiinc_t ythr_fiinc_j yint_fiinc_j ytop_fiinc_j b_fiinc_j
	sort year p f ythr_fninc_t yint_fninc_t ytop_fninc_t b_fninc_t ythr_fninc_j yint_fninc_j ytop_fninc_j b_fninc_j ythr_fiinc_t yint_fiinc_t ytop_fiinc_t b_fiinc_t ythr_fiinc_j yint_fiinc_j ytop_fiinc_j b_fiinc_j
	save StataFiles/gperc_fiscalincome_1900_2014.dta, replace
